(function(){
  'use strict';

  function ready(fn){
    if(document.readyState === 'loading'){
      document.addEventListener('DOMContentLoaded', fn);
    } else { fn(); }
  }

  function getProvider(){
    // 尝试读取当前服务器类型（按钮组/单选）
    // 常见 name 形如 ai-post[api-server-type] 或带分组：ai-post[global_tabs][api-server-type]
    var selectors = [
      'input[name*="[api-server-type]"]:checked',
      'select[name*="[api-server-type]"]',
      '[id$="api-server-type"]' // 兜底
    ];
    for (var i=0;i<selectors.length;i++){
      var el = document.querySelector(selectors[i]);
      if (!el) continue;
      if (el.tagName === 'INPUT' || el.tagName === 'SELECT'){
        var v = (el.value||'').trim();
        if (v) return v;
      }
    }
    return 'openai';
  }

  function resolveTokenKeyByProvider(provider){
    switch(String(provider||'').toLowerCase()){
      case 'custom': return 'custom_key';
      case 'doubao': return 'doubao_key';
      case 'bai-lian':
      case 'bailian': return 'bailian_key';
      case 'deepseek': return 'deepseek_key';
      case 'wenxin': return 'wenxin_key';
      default: return 'openai_key';
    }
  }

  function findSecretInputs(form){
    // 支持：input[name$="[account-secret-key]"] 或 id 以 account-secret-key 结尾
    var list = [];
    var qs = [
      'input[type="text"][name$="[account-secret-key]"]',
      'input[type="password"][name$="[account-secret-key]"]',
      'input[type="text"][id$="account-secret-key"]',
      'input[type="password"][id$="account-secret-key"]'
    ];
    qs.forEach(function(sel){
      form.querySelectorAll(sel).forEach(function(el){ list.push(el); });
    });
    // 去重
    return Array.from(new Set(list));
  }

  async function saveTokensSequentially(items){
    var results = [];
    for (var i=0;i<items.length;i++){
      var it = items[i];
      // 空值跳过
      if (!it || !it.value || !String(it.value).trim()) { continue; }
      var provider = getProvider();
      var k = resolveTokenKeyByProvider(provider);
      var v = String(it.value).trim();
      if (!window.AIPostSecure || typeof AIPostSecure.saveToken !== 'function'){
        throw new Error('AIPostSecure 工具未初始化');
      }
      await AIPostSecure.saveToken({ k: k, v: v });
      // 保存成功：清空并标记，防止进入 options.php
      try {
        it.value = '';
        // 移除 name 以防止空值字段仍被提交写入
        if (it.getAttribute('name')) {
          it.setAttribute('data-aipost-removed-name', it.getAttribute('name'));
          it.removeAttribute('name');
        }
        it.setAttribute('data-aipost-secure-saved', '1');
      } catch(e) {}
      results.push({k:k});
    }
    return results;
  }

  ready(function(){
    // 仅在设置主页面存在时运行（依赖上层按需加载）
    var forms = document.querySelectorAll('form[action*="options.php"], form');
    forms.forEach(function(form){
      if (form._aipostSecureBind) return; // 防重复
      form._aipostSecureBind = true;

      form.addEventListener('submit', function(ev){
        try {
          if (form._aipostSecureSubmitting) { return; }
          var inputs = findSecretInputs(form).filter(function(el){
            return !!el && !!String(el.value||'').trim();
          });
          if (!inputs.length) { return; }
          // 拦截默认提交 & 阻止其他监听器继续处理，避免第三方库再次劫持
          ev.preventDefault();
          try { if (typeof ev.stopImmediatePropagation === 'function') { ev.stopImmediatePropagation(); } } catch(_) {}
          (async function(){
            try {
              await saveTokensSequentially(inputs);
            } catch (e) {
              // 失败提示并终止提交（避免明文入库）
              alert('[AI-Post] 令牌保存失败：' + (e && e.message ? e.message : '未知错误'));
              return;
            }
            // 成功后再提交（避免再次拦截）
            form._aipostSecureSubmitting = true;
            try {
              // 使用原生原型的 submit，避免被第三方打桩的 form.submit() 再次触发拦截
              if (HTMLFormElement && HTMLFormElement.prototype && typeof HTMLFormElement.prototype.submit === 'function') {
                HTMLFormElement.prototype.submit.call(form);
              } else {
                form.submit();
              }
            } catch(e) {
              form._aipostSecureSubmitting = false;
            }
          })();
        } catch (e) {
          // 出异常则不拦截，走默认提交流程（但仍尽量避免泄露）
        }
      }, true);
    });
  });
})();
