<?php

/**
 * Plugin Name: 极光AI
 * Plugin URI: https://www.suqian5188.cn
 * Description: 使用AI全自动生成文章，彻底解放双手。即便你在泡妞，AI也会为你的站点不间断发布文章。开始使用：1、激活插件；2、面板配置API和KEY；3、添加任务。
 * Author: 指尖团队
 * Author URI: https://www.suqian5188.cn
 * Version: 2.8.5
 * License: GPLv3 or later
 * Text Domain: ai-post
 * Network: true
 */

namespace HaoZiTeam\AIPost;

use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

// 注册激活钩子 ▼
\register_activation_hook(__FILE__, 'HaoZiTeam\AIPost\activate_plugin');
function activate_plugin() {
    require_once \plugin_dir_path(__FILE__) . 'includes/Activator.php';
    \AIPostActivator::activate();

    // 新增：创建运行日志表
    require_once \plugin_dir_path(__FILE__) . 'Service/Features/RunLog.php';
    \HaoZiTeam\AIPost\Service\Features\RunLog::create_table();
    
    // 注册日志清理定时任务
    if (!\wp_next_scheduled('ai_post_cleanup_logs')) {
        \wp_schedule_event(time(), 'daily', 'ai_post_cleanup_logs');
    }
}

// 注册停用钩子，清理定时任务
\register_deactivation_hook(__FILE__, 'HaoZiTeam\AIPost\deactivate_plugin');
function deactivate_plugin() {
    $timestamp = \wp_next_scheduled('ai_post_cleanup_logs');
    if ($timestamp) {
        \wp_unschedule_event($timestamp, 'ai_post_cleanup_logs');
    }
}

require 'vendor/yahnis-elsts/plugin-update-checker/plugin-update-checker.php';
$MyUpdateChecker = PucFactory::buildUpdateChecker(
    'https://version.suqianweb.cn/update.json',
    __FILE__,
    'ai-post'
);

$MyUpdateChecker->addResultFilter(function($info) {
    if(isset($info->version)) {
        \update_option('ai_post_latest_version', $info->version);
    }
    return $info;
});

defined('ABSPATH') || exit;

const VERSION = '2.8.5';
const PLUGIN_FILE = __FILE__;
const PLUGIN_DIR = __DIR__;
// 新增：插件目录常量（带尾随斜杠），用于需要目录路径的场景
define('AIPOST_PLUGIN_PATH', \plugin_dir_path(__FILE__));
require_once(\plugin_dir_path(__FILE__) . 'vendor/autoload.php');
require_once \plugin_dir_path(__FILE__) . 'Inc/cgps-framework/cgps-framework.php';
require_once \plugin_dir_path(__FILE__) . 'Inc/class-article-with-pictures-api.php';
// 注册 REST 触发与自检端点（确保路由在 rest_api_init 上挂载）
require_once \plugin_dir_path(__FILE__) . 'Service/Features/CronEndpoint.php';

// 确保在命名空间外部定义常量
define('HAOZI_AIPOST_VERSION', '2.8.5');

/**
 * 数据库版本检查和升级处理
 */
function check_db_version_and_upgrade() {
    $current_version = VERSION;
    
    // 加载RunLog类
    require_once \plugin_dir_path(__FILE__) . 'Service/Features/RunLog.php';
    
    // 执行数据库升级检查
    \HaoZiTeam\AIPost\Service\Features\RunLog::check_upgrade($current_version);
    
    // 可以在这里添加其他表的升级检查...
}

// 在WordPress初始化时执行版本检查
\add_action('plugins_loaded', 'HaoZiTeam\AIPost\check_db_version_and_upgrade');

// 清理日志任务回调
function cleanup_logs_callback() {
    // 获取设置中的保留天数，默认30天
    $settings = \get_option('ai-post', []);
    $retention_days = isset($settings['log_retention_days']) ? intval($settings['log_retention_days']) : 30;
    
    // 确保值在合理范围内
    if ($retention_days < 1) {
        $retention_days = 30; // 默认至少保留30天
    }
    
    // 加载RunLog类
    require_once \plugin_dir_path(__FILE__) . 'Service/Features/RunLog.php';
    
    // 清理旧日志
    $cleaned = \HaoZiTeam\AIPost\Service\Features\RunLog::clean_old_logs($retention_days);
    \error_log("AI Post: 定时清理日志任务执行完成，清理了 {$cleaned} 条记录");
}

// 注册日志清理任务
\add_action('ai_post_cleanup_logs', 'HaoZiTeam\AIPost\cleanup_logs_callback');

// 添加修复日志表结构的函数
function fix_runlog_table() {
    // 加载RunLog类
    require_once \plugin_dir_path(__FILE__) . 'Service/Features/RunLog.php';
    
    // 尝试创建或升级表结构
    $result = \HaoZiTeam\AIPost\Service\Features\RunLog::upgrade_table_structure();
    
    // 返回结果用于AJAX响应
    return $result;
}

// 注册AJAX处理函数
\add_action('wp_ajax_ai_post_fix_runlog_table', function() {
    // 验证权限
    if (!\current_user_can('manage_options')) {
        \wp_send_json_error(['message' => '权限不足']);
        return;
    }
    
    // 执行修复
    $result = \HaoZiTeam\AIPost\fix_runlog_table();
    
    if ($result) {
        \wp_send_json_success(['message' => '日志表结构修复成功']);
    } else {
        \wp_send_json_error(['message' => '日志表结构修复失败']);
    }
});

// 在WordPress初始化时检查并修复
\add_action('admin_init', function() {
    // 只在管理员访问插件设置页面时执行
    if (isset($_GET['page']) && $_GET['page'] === 'ai-post' && isset($_GET['tab']) && $_GET['tab'] === 'log') {
        \HaoZiTeam\AIPost\fix_runlog_table();
    }
});

function load_admin_styles() {
	
    \wp_enqueue_style(
        'font-awesome', 
        'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css', 
        array(),
        null,
        'all' 
    );
}

\add_action('admin_enqueue_scripts', __NAMESPACE__ . '\\load_admin_styles');

// AJAX处理 ▼
\add_action('wp_ajax_clear_rewrite_logs', function() {
    \check_ajax_referer('clear_rewrite_logs', 'nonce');
    
    if (!\current_user_can('manage_options')) {
        \wp_send_json_error('权限不足');
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'ai_post_rewrite_logs';
    $result = $wpdb->query("TRUNCATE TABLE $table_name");
    
    if ($result !== false) {
        \wp_send_json_success('已清除所有改写记录');
    } else {
        \wp_send_json_error('清除记录失败');
    }
});

// AJAX：通信令牌（Cron/REST）管理 ▼
// 生成/刷新令牌
\add_action('wp_ajax_aipost_generate_cron_token', function() {
    \check_ajax_referer('aipost_cron_token_nonce', 'nonce');
    if (!\current_user_can('manage_options')) {
        \wp_send_json_error(['message' => '权限不足']);
    }

    if (!\function_exists('wp_generate_password')) {
        @require_once ABSPATH . 'wp-includes/pluggable.php';
    }
    $token = \wp_generate_password(24, false, false);
    \update_option('ai_post_cron_token', $token);
    \wp_send_json_success([
        'token' => $token,
        'masked' => (strlen($token) > 8) ? (substr($token, 0, 4) . str_repeat('*', max(0, strlen($token) - 8)) . substr($token, -4)) : $token,
    ]);
});

// 获取令牌（明文，仅一次性展示，前端自行控制不持久化）
\add_action('wp_ajax_aipost_get_cron_token', function() {
    \check_ajax_referer('aipost_cron_token_nonce', 'nonce');
    if (!\current_user_can('manage_options')) {
        \wp_send_json_error(['message' => '权限不足']);
    }
    $token = (string) \get_option('ai_post_cron_token', '');
    if ($token === '') {
        \wp_send_json_error(['message' => '尚未生成']);
    }
    \wp_send_json_success([
        'token' => $token,
        'masked' => (strlen($token) > 8) ? (substr($token, 0, 4) . str_repeat('*', max(0, strlen($token) - 8)) . substr($token, -4)) : $token,
    ]);
});

// 清空令牌（禁用外部触发入口）
\add_action('wp_ajax_aipost_clear_cron_token', function() {
    \check_ajax_referer('aipost_cron_token_nonce', 'nonce');
    if (!\current_user_can('manage_options')) {
        \wp_send_json_error(['message' => '权限不足']);
    }
    \delete_option('ai_post_cron_token');
    \wp_send_json_success(['message' => '已清空通信令牌']);
});

// 添加网络管理菜单支持
function add_network_admin_menu() {
    if (is_multisite() && is_network_admin()) {
        add_menu_page(
            '极光AI设置', 
            '极光AI', 
            'manage_network_options',
            'ai-post-network', 
            'HaoZiTeam\AIPost\network_admin_page',
            'dashicons-admin-generic',
            81
        );
    }
}
add_action('network_admin_menu', 'HaoZiTeam\AIPost\add_network_admin_menu');

// 网络管理页面回调
function network_admin_page() {
    echo '<div class="wrap">';
    echo '<h1>极光AI - 网络设置</h1>';
    echo '<p>您正在查看网络管理页面。请前往各个站点的设置页面配置极光AI。</p>';
    echo '<p>在多站点模式下，每个子站点需要单独配置极光AI。</p>';
    
    // 显示所有站点及其链接
    $sites = get_sites();
    if (!empty($sites)) {
        echo '<h2>站点列表</h2>';
        echo '<ul>';
        foreach ($sites as $site) {
            $site_id = $site->blog_id;
            $site_url = get_site_url($site_id);
            $site_name = get_blog_details($site_id)->blogname;
            $admin_url = get_admin_url($site_id, 'options-general.php?page=ai-post');
            
            echo '<li>';
            echo '<strong>' . esc_html($site_name) . '</strong> - ';
            echo '<a href="' . esc_url($admin_url) . '" target="_blank">配置极光AI</a>';
            echo '</li>';
        }
        echo '</ul>';
    }
    
    echo '</div>';
}

new Plugin(); 