<?php

namespace HaoZiTeam\AIPost\Service\Settings\Traits;

defined('ABSPATH') || exit;

trait TraitImageGenCheck
{
    // AJAX：OpenAI 兼容端点连通性自检（/v1/models）
    public function ajax_connectivity_check(): void
    {
        \check_ajax_referer('ai_post_connectivity_check');
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => '权限不足'], 403);
        }

        $opts = \get_option('ai-post', []);
        if (!is_array($opts)) { $opts = []; }

        $type = $opts['api-server-type'] ?? 'custom';
        if (!in_array($type, ['openai','custom','', null], true)) {
            \wp_send_json_error(['message' => '当前仅支持 OpenAI 兼容/自定义端点的自检']);
        }

        $server = $opts['openai-api-server'] ?? 'https://sk.slapi.cn/';
        $apiKey = '';
        if (!empty($opts['openai-accounts']) && is_array($opts['openai-accounts'])) {
            $accounts = $opts['openai-accounts'];
            if (!empty($accounts)) {
                $idx = rand(1, count($accounts));
                $acc = $accounts[$idx - 1];
                $apiKey = trim((string)($acc['account-secret-key'] ?? ''));
            }
        }
        if (!$apiKey && !empty($opts['cardPwd'])) { $apiKey = trim((string)$opts['cardPwd']); }
        if (!$apiKey && !empty($opts['cardNum'])) { $apiKey = trim((string)$opts['cardNum']); }

        if (!$apiKey) {
            \wp_send_json_error(['message' => '未找到可用的 API Key，请在设置中填写 OpenAI 兼容密钥']);
        }

        // 规范 server：确保以单一 /v1/ 结尾
        $server = trim((string)$server);
        $server = rtrim($server, "/\t\n\r ");
        if (!preg_match('#/v1/?$#', $server)) {
            $server = rtrim($server, '/') . '/v1';
        }
        if (substr($server, -1) !== '/') { $server .= '/'; }

        $url = $server . 'models';
        $args = [
            'timeout' => 30,
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type'  => 'application/json',
            ],
        ];

        $start = microtime(true);
        $resp = \wp_remote_get($url, $args);
        $duration = (int) round((microtime(true) - $start) * 1000);

        if (is_wp_error($resp)) {
            \wp_send_json_error(['message' => '请求失败：' . $resp->get_error_message()]);
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = (string) wp_remote_retrieve_body($resp);
        $snippet = mb_substr($body, 0, 500, 'UTF-8');

        \wp_send_json_success([
            'status' => $code,
            'url' => $url,
            'duration_ms' => $duration,
            'snippet' => $snippet,
        ]);
    }

    // AJAX：火山 Ark 文生图参数自检（不发起外部请求，仅校验并回显即将发送的负载）
    public function ajax_volc_imagegen_check(): void
    {
        \check_ajax_referer('ai_post_volc_imagegen_check');
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => '权限不足'], 403);
        }

        $opts = \get_option('ai-post', []);
        if (!is_array($opts)) { $opts = []; }

        $model = (string) ($opts['doubao-imagegen-model'] ?? 'doubao-seedream-3-0-t2i-250415');
        $size  = '1024x768';
        $watermark = (bool) ($opts['doubao-imagegen-watermark'] ?? false);

        $hasKey = false;
        if (!empty($opts['doubao-accounts']) && is_array($opts['doubao-accounts'])) {
            foreach ($opts['doubao-accounts'] as $acc) {
                $k = isset($acc['account-secret-key']) ? trim((string)$acc['account-secret-key']) : '';
                if ($k !== '') { $hasKey = true; break; }
            }
        }

        $endpoint = 'https://ark.cn-beijing.volces.com/api/v3/images/generations';
        $payload = [
            'model' => $model,
            'prompt' => 'Sanity check prompt (will not be sent).',
            'response_format' => 'url',
            'size' => $size,
            'watermark' => $watermark,
        ];

        \wp_send_json_success([
            'endpoint' => $endpoint,
            'model' => $model,
            'size' => $size,
            'watermark' => $watermark,
            'has_key' => $hasKey,
            'payload' => $payload,
            'payload_pretty' => wp_json_encode($payload, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE),
        ]);
    }

    // AJAX：阿里云（DashScope）文生图参数自检（不发起外部请求，仅校验并回显即将发送的负载）
    public function ajax_aliyun_imagegen_check(): void
    {
        \check_ajax_referer('ai_post_aliyun_imagegen_check');
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => '权限不足'], 403);
        }

        $opts = \get_option('ai-post', []);
        if (!is_array($opts)) { $opts = []; }

        $model = (string)($opts['aliyun-imagegen-model'] ?? 'wan2.2-t2i-flash');
        $size = ($model === 'qwen-image') ? '1328*1328' : '1024*1024';
        $n     = 1;
        $promptExtend = (bool)($opts['aliyun-imagegen-prompt-extend'] ?? true);
        $watermark    = (bool)($opts['aliyun-imagegen-watermark'] ?? true);
        $style        = (string)($opts['aliyun-imagegen-style'] ?? '<auto>');

        $hasKey = false;
        if (!empty($opts['bai-lian-accounts']) && is_array($opts['bai-lian-accounts'])) {
            foreach ($opts['bai-lian-accounts'] as $acc) {
                $k = isset($acc['account-secret-key']) ? trim((string)$acc['account-secret-key']) : '';
                if ($k !== '') { $hasKey = true; break; }
            }
        }

        $endpoint = 'https://dashscope.aliyuncs.com/api/v1/services/aigc/text2image/image-synthesis';
        $payload = [
            'model' => $model,
            'input' => [
                'prompt' => 'Sanity check prompt (will not be sent).',
            ],
            'parameters' => array_filter([
                'size' => $size,
                'n'    => $n,
                'prompt_extend' => ($model === 'qwen-image') ? $promptExtend : null,
                'watermark'     => ($model === 'qwen-image') ? $watermark : null,
                'style' => ($model === 'wanx-v1') ? $style : null,
            ], function($v){ return $v !== null; })
        ];

        \wp_send_json_success([
            'endpoint' => $endpoint,
            'model' => $model,
            'size' => $size,
            'n' => $n,
            'prompt_extend' => $promptExtend,
            'watermark' => $watermark,
            'style' => $style,
            'has_key' => $hasKey,
            'payload' => $payload,
            'payload_pretty' => wp_json_encode($payload, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE),
        ]);
    }

    // AJAX：阿里云（DashScope）文生图外网连通性自检（真调用，最小尺寸）
    public function ajax_aliyun_imagegen_connectivity(): void
    {
        \check_ajax_referer('ai_post_aliyun_imagegen_connectivity');
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => '权限不足'], 403);
        }

        $opts = \get_option('ai-post', []);
        if (!is_array($opts)) { $opts = []; }

        $apiKey = '';
        if (!empty($opts['bai-lian-accounts']) && is_array($opts['bai-lian-accounts'])) {
            foreach ($opts['bai-lian-accounts'] as $acc) {
                $k = isset($acc['account-secret-key']) ? trim((string)$acc['account-secret-key']) : '';
                if ($k !== '') { $apiKey = $k; break; }
            }
        }
        if ($apiKey === '') {
            \wp_send_json_error(['message' => '未检测到可用的阿里百炼 API Key（请在“阿里百炼 令牌列表”中配置 account-secret-key）']);
        }

        $model = (string) ($opts['aliyun-imagegen-model'] ?? 'wan2.2-t2i-flash');
        $size = ($model === 'qwen-image') ? '1328*1328' : '1024*1024';
        $n = 1;
        $promptExtend = (bool)($opts['aliyun-imagegen-prompt-extend'] ?? true);
        $watermark    = (bool)($opts['aliyun-imagegen-watermark'] ?? true);
        $style        = (string)($opts['aliyun-imagegen-style'] ?? '<auto>');

        $endpoint = 'https://dashscope.aliyuncs.com/api/v1/services/aigc/text2image/image-synthesis';
        $payload = [
            'model' => $model,
            'input' => [
                'prompt' => 'Connectivity test from AI-Post plugin. Please generate a simple abstract background. This is a minimal test call.'
            ],
            'parameters' => array_filter([
                'size' => $size,
                'n'    => $n,
                'prompt_extend' => ($model === 'qwen-image') ? $promptExtend : null,
                'watermark'     => ($model === 'qwen-image') ? $watermark : null,
                'style' => ($model === 'wanx-v1') ? $style : null,
            ], function($v){ return $v !== null; })
        ];
        $json = \wp_json_encode($payload);

        $headers = [
            'Content-Type: application/json',
            'Accept: application/json',
            'Authorization: Bearer ' . $apiKey,
            'X-DashScope-Async: enable',
        ];

        // Trait 目录位于 Service/Settings/Traits，插件根需上溯三级
        $pluginRoot = \dirname(__DIR__, 3);
        $caFile = $pluginRoot . DIRECTORY_SEPARATOR . 'resources' . DIRECTORY_SEPARATOR . 'cacert.pem';

        $ch = \curl_init($endpoint);
        if ($ch === false) {
            \wp_send_json_error(['message' => 'cURL 初始化失败']);
        }
        \curl_setopt($ch, CURLOPT_POST, true);
        \curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        \curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
        \curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        \curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        \curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        \curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        if (@file_exists($caFile)) {
            \curl_setopt($ch, CURLOPT_CAINFO, $caFile);
        }

        $start = microtime(true);
        $response = \curl_exec($ch);
        $duration = (int) ((microtime(true) - $start) * 1000);
        $errno = \curl_errno($ch);
        $error = \curl_error($ch);
        $httpCode = (int) \curl_getinfo($ch, CURLINFO_HTTP_CODE);
        \curl_close($ch);

        if ($errno) {
            \wp_send_json_error(['message' => 'cURL 错误: ' . $errno . ' ' . $error]);
        }

        $snippet = is_string($response) ? mb_substr($response, 0, 1000) : '';
        $data = json_decode((string)$response, true);
        if (!is_array($data)) {
            \wp_send_json_error([
                'message' => '响应解析失败',
                'status' => $httpCode,
                'duration_ms' => $duration,
                'snippet' => $snippet,
            ]);
        }

        if ($httpCode < 200 || $httpCode >= 300) {
            $msg = $data['error']['message'] ?? ('HTTP ' . $httpCode);
            \wp_send_json_error([
                'message' => 'HTTP错误: ' . $msg,
                'status' => $httpCode,
                'duration_ms' => $duration,
                'snippet' => $snippet,
            ]);
        }

        \wp_send_json_success([
            'status' => $httpCode,
            'duration_ms' => $duration,
            'snippet' => $snippet,
        ]);
    }

    // AJAX：ImageGenerations 参数自检（不发起外部请求，仅校验并回显即将发送的负载）
    public function ajax_imagegen_check(): void
    {
        \check_ajax_referer('ai_post_imagegen_check');
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => '权限不足'], 403);
        }

        $opts = \get_option('ai-post', []);
        if (!is_array($opts)) { $opts = []; }

        $model = (string) ($opts['imagegen-model'] ?? '');
        $size  = (string) ($opts['imagegen-size'] ?? '1024x768');
        if ($size === 'custom') {
            $w = (int) ($opts['imagegen-custom-width'] ?? 1024);
            $h = (int) ($opts['imagegen-custom-height'] ?? 768);
            $size = $w . 'x' . $h;
        }
        $watermark = (bool) ($opts['imagegen-watermark'] ?? false);

        $hasKey = false;
        if (!empty($opts['imagegen-accounts']) && is_array($opts['imagegen-accounts'])) {
            foreach ($opts['imagegen-accounts'] as $acc) {
                $k = isset($acc['account-secret-key']) ? trim((string)$acc['account-secret-key']) : '';
                if ($k !== '') { $hasKey = true; break; }
            }
        }

        $endpoint = 'https://api.openai.com/v1/images/generations';
        $payload = [
            'model' => $model,
            'prompt' => 'Sanity check prompt (will not be sent).',
            'response_format' => 'url',
            'size' => $size,
            'watermark' => $watermark,
        ];

        \wp_send_json_success([
            'endpoint' => $endpoint,
            'model' => $model,
            'size' => $size,
            'watermark' => $watermark,
            'has_key' => $hasKey,
            'payload' => $payload,
            'payload_pretty' => wp_json_encode($payload, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE),
        ]);
    }
}
