<?php

namespace HaoZiTeam\AIPost\Service\Settings\Traits;

defined('ABSPATH') || exit;

trait TraitExportImport
{
    // 处理导出设置的 AJAX 请求
    public function handle_export_settings(): void
    {
        // 验证 Nonce
        if (!isset($_POST['nonce']) || !\wp_verify_nonce($_POST['nonce'], 'ai_post_export_settings_nonce')) {
            \wp_send_json_error(['message' => 'Nonce 验证失败'], 403);
            return;
        }

        // 检查用户权限
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => '权限不足'], 403);
            return;
        }

        // 获取插件设置
        $settings = \get_option($this->prefix);

        if ($settings === false) {
            \wp_send_json_error(['message' => '无法获取设置'], 500);
            return;
        }

        // 排除授权相关的参数
        if (isset($settings['cardNum'])) {
            unset($settings['cardNum']);
        }
        if (isset($settings['cardPwd'])) {
            unset($settings['cardPwd']);
        }

        // 排除 API 令牌列表
        if (isset($settings['openai-accounts'])) {
            unset($settings['openai-accounts']);
        }
        if (isset($settings['deepseek-accounts'])) {
            unset($settings['deepseek-accounts']);
        }
        if (isset($settings['doubao-accounts'])) {
            unset($settings['doubao-accounts']);
        }

        // 将过滤后的设置编码为 JSON
        $settings_json = \wp_json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

        if ($settings_json === false) {
             \wp_send_json_error(['message' => 'JSON 编码失败'], 500);
            return;
        }

        // 发送成功响应
        \wp_send_json_success(['settings_json' => $settings_json]);
    }

    // 处理导入设置的 AJAX 请求
    public function handle_import_settings(): void
    {
        // 验证 Nonce
        if (!isset($_POST['nonce']) || !\wp_verify_nonce($_POST['nonce'], 'ai_post_import_settings_nonce')) {
            \wp_send_json_error(['message' => 'Nonce 验证失败'], 403);
            return;
        }

        // 检查用户权限
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => '权限不足'], 403);
            return;
        }

        // 获取并验证 JSON 数据
        if (!isset($_POST['settings_json']) || empty($_POST['settings_json'])) {
            \wp_send_json_error(['message' => '未提供导入数据'], 400);
            return;
        }

        $settings_json = \wp_unslash($_POST['settings_json']); // 移除反斜杠
        $imported_settings = json_decode($settings_json, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            \wp_send_json_error(['message' => '无效的 JSON 格式: ' . json_last_error_msg()], 400);
            return;
        }

        if (!is_array($imported_settings)) {
             \wp_send_json_error(['message' => '导入的数据必须是有效的设置数组'], 400);
            return;
        }

        // 获取当前的设置
        $current_settings = \get_option($this->prefix, []); // 第二个参数确保在选项不存在时返回空数组
        if (!is_array($current_settings)) { // 做个健壮性检查
            $current_settings = [];
        }

        // 合并导入的设置到当前设置，导入的设置会覆盖同名的现有设置
        // 但不会删除当前设置中存在而导入设置中不存在的键（如授权信息）
        $merged_settings = array_replace_recursive($current_settings, $imported_settings);

        // 更新设置
        $updated = \update_option($this->prefix, $merged_settings);

        if ($updated) {
            \wp_send_json_success(['message' => '设置已成功导入']);
            // 检查一致性（保留原注释与逻辑）
            $check_settings = \get_option($this->prefix);
            if ($check_settings === $merged_settings) {
                 \wp_send_json_success(['message' => '设置已成功导入（部分设置可能未改变）']);
            } else {
                \wp_send_json_error(['message' => '无法更新设置，请稍后重试'], 500);
            }
        } else {
            // update_option 在值未改变时也会返回 false，需要区分这种情况和真正错误
            $check_settings = \get_option($this->prefix);
            if ($check_settings === $merged_settings) {
                 \wp_send_json_success(['message' => '设置已成功导入（部分设置可能未改变）']);
            } else {
                \wp_send_json_error(['message' => '无法更新设置，请稍后重试'], 500);
            }
        }
    }
}
