<?php
// Service/Features/StyleUtils.php
// 该文件主要负责提供样式（CSS）输出相关的辅助功能。

namespace HaoZiTeam\AIPost\Service\Features;

defined('ABSPATH') || exit;

class StyleUtils
{
    /**
     * 输出全局自定义 CSS 和文章目录 (TOC) 的 CSS。
     *
     * @param array $settings 插件的设置数组。
     * @return void
     */
    public static function output_global_styles($settings)
    {
        $global_css = $settings['global_tabs']['global_custom_css'] ?? '';
        if (!empty($global_css)) {
            echo "\n<!-- 全局自定义CSS -->\n<style>\n" . esc_html($global_css) . "\n</style>\n";
        }

        // 新增：为文章目录添加基础CSS
        if ($settings['global_tabs']['enable-toc'] ?? false) {
            echo "\n<!-- AI Post TOC CSS -->\n<style>\n";
            echo ".ai-post-toc .toc-title { cursor: pointer; }\n";
            echo ".ai-post-toc .toc-list { list-style: none; padding-left: 0; margin-bottom: 0; overflow: hidden; transition: max-height 0.3s ease-out; }\n";
            echo ".ai-post-toc.toc-collapsed .toc-list { max-height: 0; }\n";
            echo ".ai-post-toc.toc-expanded .toc-list { max-height: 1000px; /* Or a large enough value */ }\n"; // 展开时的高度
            echo ".ai-post-toc .toc-toggle-icon { display: inline-block; margin-left: 5px; transition: transform 0.3s ease; font-size: 12px; vertical-align: middle; }\n";
            echo ".ai-post-toc.toc-collapsed .toc-toggle-icon { transform: rotate(-90deg); }\n";
            echo ".ai-post-toc ul ul { padding-left: 20px; } /* Add padding for nested lists */ \n";
            echo ".ai-post-toc li { margin-bottom: 5px; } /* Add some space between list items */ \n";
            // 新增：控制 Open/Close 文本显示
            echo ".ai-post-toc.toc-expanded .toc-text-open { display: none; }\n";
            echo ".ai-post-toc.toc-collapsed .toc-text-close { display: none; }\n";
            echo ".ai-post-toc .toc-status-text { display: inline-block; vertical-align: middle; }\n"; // 确保文本和图标对齐
            echo "</style>\n";
        }
    }

    /**
     * 输出文章目录 (TOC) 的 JavaScript。
     *
     * @param array $settings 插件的设置数组。
     * @return void
     */
    public static function output_toc_script($settings)
    {
        // 检查是否启用了TOC功能，并且只在单个文章页面输出脚本
        if (!empty($settings['global_tabs']['enable-toc']) && is_single()) {
            // 获取TOC设置
            $toc_title_text = esc_js($settings['global_tabs']['toc-title-text'] ?? '文章目录');
            $default_state = $settings['global_tabs']['toc-default-state'] ?? 'expanded';
            $initial_class = ($default_state === 'collapsed') ? 'toc-collapsed' : 'toc-expanded';

            // JavaScript for TOC toggle and smooth scroll
            ?>
            <script type="text/javascript">
            document.addEventListener("DOMContentLoaded", function() {
                const tocDiv = document.querySelector(".ai-post-toc");
                if (!tocDiv) return;

                // Add initial state class based on settings
                tocDiv.classList.add('<?php echo $initial_class; ?>');

                const tocTitle = tocDiv.querySelector(".toc-title");
                const tocList = tocDiv.querySelector(".toc-list");
                const tocLinks = tocDiv.querySelectorAll(".toc-link");

                if (tocTitle && tocList) {
                    tocTitle.addEventListener("click", function() {
                        tocDiv.classList.toggle("toc-collapsed");
                        tocDiv.classList.toggle("toc-expanded");
                    });
                }

                if (tocLinks) {
                    tocLinks.forEach(function(link) {
                        link.addEventListener("click", function(e) {
                            e.preventDefault();
                            const targetId = this.getAttribute("href").substring(1);
                            const targetElement = document.getElementById(targetId);
                            if (targetElement) {
                                const offset = 80; 
                                const bodyRect = document.body.getBoundingClientRect().top;
                                const elementRect = targetElement.getBoundingClientRect().top;
                                const elementPosition = elementRect - bodyRect;
                                const offsetPosition = elementPosition - offset;

                                window.scrollTo({
                                    top: offsetPosition,
                                    behavior: "smooth"
                                });

                                if (tocDiv.classList.contains("toc-collapsed")) {
                                    tocDiv.classList.remove("toc-collapsed");
                                    tocDiv.classList.add("toc-expanded");
                                }
                            } else {
                                window.location.href = this.getAttribute("href");
                            }
                        });
                    });
                }
            });
            </script>
            <?php
        }
    }
} 