<?php
namespace HaoZiTeam\AIPost\Service\Features;

defined('ABSPATH') || exit;

class RunLog
{
    private static $table = '';
    
    // 日志级别常量
    const LOG_LEVEL_INFO = 0;
    const LOG_LEVEL_WARNING = 1;
    const LOG_LEVEL_ERROR = 2;
    
    /**
     * 获取日志级别名称
     * 
     * @param int $level 日志级别
     * @return string 日志级别名称
     */
    public static function get_level_name($level)
    {
        switch ($level) {
            case self::LOG_LEVEL_WARNING:
                return '警告';
            case self::LOG_LEVEL_ERROR:
                return '错误';
            case self::LOG_LEVEL_INFO:
            default:
                return '信息';
        }
    }
    
    /**
     * 获取日志级别颜色
     * 
     * @param int $level 日志级别
     * @return string 日志级别CSS颜色
     */
    public static function get_level_color($level)
    {
        switch ($level) {
            case self::LOG_LEVEL_WARNING:
                return '#f0b849'; // 黄色警告
            case self::LOG_LEVEL_ERROR:
                return '#d63638'; // 红色错误
            case self::LOG_LEVEL_INFO:
            default:
                return '#2271b1'; // 蓝色信息
        }
    }
    
    /**
     * 获取表名
     *
     * @return string
     */
    public static function get_table_name()
    {
        global $wpdb;
        if (!self::$table) {
            self::$table = $wpdb->prefix . 'ai_post_run_logs';
        }
        return self::$table;
    }

    /**
     * 建表方法，激活插件或升级时调用
     *
     * @return bool 是否成功创建表
     */
    public static function create_table()
    {
        global $wpdb;
        $table = self::get_table_name();
        $charset_collate = $wpdb->get_charset_collate();
        
        // 使用优化后的表结构
        $sql = "CREATE TABLE IF NOT EXISTS $table (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            instruction TEXT NULL,  -- 改为TEXT类型，不限制长度
            title VARCHAR(255) NOT NULL,
            original_title VARCHAR(255) NULL,  -- 新增字段：原始标题
            created_at DATETIME NOT NULL,
            status VARCHAR(32) NOT NULL,
            extra TEXT NULL,
            is_rewrite VARCHAR(20) NULL,
            log_level TINYINT DEFAULT 0, -- 新增日志级别字段
            PRIMARY KEY (id),
            INDEX idx_created_at (created_at), -- 添加时间索引
            INDEX idx_status (status), -- 添加状态索引
            INDEX idx_title (title(50)) -- 添加标题索引
        ) $charset_collate;";
        
        // 使用全局命名空间引用WordPress函数
        if (file_exists(ABSPATH . 'wp-admin/includes/upgrade.php')) {
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        } else {
            // 如果无法加载upgrade.php，尝试直接执行SQL
            $wpdb->query($sql);
            \error_log("AI Post: 无法加载dbDelta函数，直接执行SQL创建表");
        }
        
        // 记录表创建日志
        \error_log("AI Post: RunLog表创建/更新完成");
        
        // 检查表是否正确创建和包含必要字段
        $check_table = $wpdb->get_var("SHOW TABLES LIKE '$table'");
        if ($check_table !== $table) {
            \error_log("AI Post Error: RunLog表创建失败");
            return false;
        }
        
        // 检查log_level字段是否存在
        $check_column = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'log_level'");
        if (empty($check_column)) {
            \error_log("AI Post Warning: log_level字段未创建，尝试手动添加");
            $wpdb->query("ALTER TABLE $table ADD COLUMN log_level TINYINT DEFAULT 0 AFTER is_rewrite");
            
            // 再次检查是否添加成功
            $check_column = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'log_level'");
            if (empty($check_column)) {
                \error_log("AI Post Error: 手动添加log_level字段失败");
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * 升级数据库表结构
     * 
     * @return bool 是否成功升级
     */
    public static function upgrade_table_structure()
    {
        global $wpdb;
        $table = self::get_table_name();
        $success = true;
        
        // 检查表是否存在
        if (!self::table_exists()) {
            // 如果表不存在，直接创建新表
            return self::create_table();
        }
        
        // 开始升级现有表结构
        try {
            // 1. 修改instruction字段类型为TEXT
            $wpdb->query("ALTER TABLE $table MODIFY instruction TEXT NULL");
            
            // 2. 添加log_level字段（如果不存在）
            $check_column = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'log_level'");
            if (empty($check_column)) {
                $result = $wpdb->query("ALTER TABLE $table ADD COLUMN log_level TINYINT DEFAULT 0 AFTER is_rewrite");
                if ($result === false) {
                    \error_log("AI Post Error: 添加log_level字段失败: " . $wpdb->last_error);
                    $success = false;
                } else {
                    \error_log("AI Post: 添加log_level字段成功");
                }
            } else {
                \error_log("AI Post: log_level字段已存在，无需添加");
            }
            
            // 3. 添加original_title字段（如果不存在）
            $check_column = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'original_title'");
            if (empty($check_column)) {
                $result = $wpdb->query("ALTER TABLE $table ADD COLUMN original_title VARCHAR(255) NULL AFTER title");
                if ($result === false) {
                    \error_log("AI Post Error: 添加original_title字段失败: " . $wpdb->last_error);
                    $success = false;
                } else {
                    \error_log("AI Post: 添加original_title字段成功");
                }
            } else {
                \error_log("AI Post: original_title字段已存在，无需添加");
            }
            
            // 4. 添加索引（先检查索引是否存在）
            $indexes = $wpdb->get_results("SHOW INDEX FROM $table");
            $existing_indexes = [];
            
            foreach ($indexes as $index) {
                $existing_indexes[] = $index->Key_name;
            }
            
            // 添加created_at索引
            if (!in_array('idx_created_at', $existing_indexes)) {
                $wpdb->query("CREATE INDEX idx_created_at ON $table (created_at)");
                \error_log("AI Post: 添加created_at索引成功");
            }
            
            // 添加status索引
            if (!in_array('idx_status', $existing_indexes)) {
                $wpdb->query("CREATE INDEX idx_status ON $table (status)");
                \error_log("AI Post: 添加status索引成功");
            }
            
            // 添加title索引
            if (!in_array('idx_title', $existing_indexes)) {
                $wpdb->query("CREATE INDEX idx_title ON $table (title(50))");
                \error_log("AI Post: 添加title索引成功");
            }
            
            \error_log("AI Post: 数据库表结构升级成功");
            return $success;
            
        } catch (\Exception $e) {
            \error_log("AI Post: 数据库表结构升级失败: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 检查表是否存在
     *
     * @return bool
     */
    public static function table_exists()
    {
        global $wpdb;
        $table = self::get_table_name();
        return $wpdb->get_var("SHOW TABLES LIKE '$table'") === $table;
    }
    
    /**
     * 检查版本并执行必要的升级
     *
     * @param string $current_version 当前插件版本
     */
    public static function check_upgrade($current_version)
    {
        // 获取数据库版本
        $db_version = \get_option('ai_post_db_version', '0.0.0');
        
        // 如果数据库版本低于当前版本，执行升级
        if (version_compare($db_version, $current_version, '<')) {
            self::upgrade_table_structure();
            
            // 更新数据库版本号
            \update_option('ai_post_db_version', $current_version);
            
            \error_log("AI Post: 数据库版本已从 {$db_version} 升级到 {$current_version}");
        }
    }
    
    /**
     * 添加日志
     *
     * @param string $instruction 指令
     * @param string $title 标题
     * @param string $status 状态
     * @param string $extra 额外信息
     * @param string|null $is_rewrite 是否为改写
     * @param int $log_level 日志级别
     * @param string|null $original_title 原始标题（改写前）
     */
    public static function add($instruction, $title, $status, $extra = '', $is_rewrite = null, $log_level = 0, $original_title = null)
    {
        global $wpdb;
        $table = self::get_table_name();
        
        // 添加详细调试日志
        \error_log("AI Post Debug: 尝试记录日志 - 标题: {$title}, 状态: {$status}, 是否改写: " . ($is_rewrite ?: '否') . ", 级别: {$log_level}" . ($original_title ? ", 原标题: {$original_title}" : ""));
        
        // 检查表是否存在，不存在则创建
        if (!self::table_exists()) {
            \error_log("AI Post Debug: 日志表不存在，尝试创建");
            $created = self::create_table();
            if (!$created) {
                \error_log("AI Post Debug: 创建日志表失败");
                return;
            }
        }
        
        // 检查表结构
        $check_column = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'log_level'");
        if (empty($check_column)) {
            \error_log("AI Post Debug: 日志表缺少log_level字段，尝试升级表结构");
            self::upgrade_table_structure();
            
            // 再次检查，如果还是不存在，则尝试直接添加
            $check_column = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'log_level'");
            if (empty($check_column)) {
                \error_log("AI Post Debug: 升级后仍缺少log_level字段，尝试直接添加");
                $wpdb->query("ALTER TABLE $table ADD COLUMN log_level TINYINT DEFAULT 0 AFTER is_rewrite");
            }
        }
        
        // 检查标题是否为空
        if (empty($title)) {
            \error_log("AI Post Debug: 标题为空，跳过日志记录");
            return;
        }
        
        // 记录插入时间
        $current_time = current_time('mysql');
        
        try {
            // 如果是改写记录（包括"重复改写"、"超长改写"和"空标题改写"）
            if (!empty($is_rewrite)) {
                // 改写前先删除同标题的所有记录（不仅限于一分钟内的）
                $wpdb->query($wpdb->prepare(
                    "DELETE FROM $table WHERE title = %s",
                    $title
                ));
                
                // 插入新的改写记录
                $data = [
                    'instruction' => $instruction, 
                    'title' => $title,
                    'original_title' => $original_title, // 添加原始标题
                    'created_at' => $current_time,
                    'status' => $status,
                    'extra' => $extra,
                    'is_rewrite' => $is_rewrite,
                    'log_level' => $log_level // 添加日志级别
                ];
                
                \error_log("AI Post Debug: 尝试插入改写日志 - " . json_encode($data, JSON_UNESCAPED_UNICODE));
                
                $result = $wpdb->insert($table, $data);
                
                if ($result) {
                    \error_log("AI Post Debug: 成功记录改写日志 - 指令: {$instruction}, 标题: {$title}" . ($original_title ? ", 原标题: {$original_title}" : ""));
                } else {
                    \error_log("AI Post Debug: 记录改写日志失败 - 指令: {$instruction}, 标题: {$title}, 错误: " . $wpdb->last_error);
                }
            } else {
                // 非改写记录，检查是否存在相同标题的记录
                $existing_count = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM $table WHERE title = %s AND created_at > DATE_SUB(%s, INTERVAL 1 MINUTE)",
                    $title, $current_time
                ));
                
                // 如果不存在相同标题的记录，才插入
                if ($existing_count == 0) {
                    $data = [
                        'instruction' => $instruction,
                        'title' => $title,
                        'original_title' => $original_title, // 添加原始标题
                        'created_at' => $current_time,
                        'status' => $status,
                        'extra' => $extra,
                        'is_rewrite' => $is_rewrite,
                        'log_level' => $log_level // 添加日志级别
                    ];
                    
                    \error_log("AI Post Debug: 尝试插入普通日志 - " . json_encode($data, JSON_UNESCAPED_UNICODE));
                    
                    $result = $wpdb->insert($table, $data);
                    
                    if ($result) {
                        \error_log("AI Post Debug: 成功记录普通日志 - 指令: {$instruction}, 标题: {$title}");
                    } else {
                        \error_log("AI Post Debug: 记录普通日志失败 - 指令: {$instruction}, 标题: {$title}, 错误: " . $wpdb->last_error);
                    }
                } else {
                    \error_log("AI Post Debug: 跳过重复日志 - 指令: {$instruction}, 标题: {$title}");
                }
            }
        } catch (\Exception $e) {
            \error_log("AI Post Debug: 记录日志异常 - " . $e->getMessage());
        }
    }

    /**
     * 读取日志，支持分页
     *
     * @param int $paged 页码
     * @param int $per_page 每页数量
     * @return array
     */
    public static function get_logs($paged = 1, $per_page = 20)
    {
        global $wpdb;
        $table = self::get_table_name();
        
        // 检查表是否存在
        if (!self::table_exists()) {
            return [
                'logs' => [],
                'total' => 0
            ];
        }
        
        $offset = ($paged - 1) * $per_page;
        $logs = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table ORDER BY id DESC LIMIT %d OFFSET %d",
            $per_page, $offset
        ), 'ARRAY_A');
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $table");
        return [
            'logs' => $logs,
            'total' => intval($total)
        ];
    }
    
    /**
     * 获取筛选后的日志
     *
     * @param array $filters 筛选条件
     * @param int $paged 页码
     * @param int $per_page 每页数量
     * @return array
     */
    public static function get_filtered_logs($filters = [], $paged = 1, $per_page = 20)
    {
        global $wpdb;
        $table = self::get_table_name();
        
        $where = [];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = %s";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['is_rewrite'])) {
            $where[] = "is_rewrite = %s";
            $params[] = $filters['is_rewrite'];
        }
        
        if (!empty($filters['date_from'])) {
            $where[] = "created_at >= %s";
            $params[] = $filters['date_from'];
        }
        
        if (!empty($filters['date_to'])) {
            $where[] = "created_at <= %s";
            $params[] = $filters['date_to'];
        }
        
        if (!empty($filters['search'])) {
            $where[] = "(title LIKE %s OR instruction LIKE %s)";
            $search = '%' . $wpdb->esc_like($filters['search']) . '%';
            $params[] = $search;
            $params[] = $search;
        }
        
        if (!empty($filters['log_level'])) {
            $where[] = "log_level = %d";
            $params[] = intval($filters['log_level']);
        }
        
        $where_sql = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";
        
        // 计算总数
        $count_query = "SELECT COUNT(*) FROM $table $where_sql";
        $prepared_count_query = !empty($params) ? $wpdb->prepare($count_query, $params) : $count_query;
        $total = $wpdb->get_var($prepared_count_query);
        
        // 获取分页数据
        $offset = ($paged - 1) * $per_page;
        $query_params = array_merge($params, [$per_page, $offset]);
        
        $query = "SELECT * FROM $table $where_sql ORDER BY id DESC LIMIT %d OFFSET %d";
        $prepared_query = !empty($params) ? $wpdb->prepare($query, $query_params) : $wpdb->prepare($query, [$per_page, $offset]);
        
        $logs = $wpdb->get_results($prepared_query, 'ARRAY_A');
        
        return [
            'logs' => $logs,
            'total' => intval($total)
        ];
    }
    
    /**
     * 清空所有日志
     *
     * @return bool
     */
    public static function clear()
    {
        global $wpdb;
        $table = self::get_table_name();
        
        if (!self::table_exists()) {
            return false;
        }
        
        return $wpdb->query("TRUNCATE TABLE $table") !== false;
    }
    
    /**
     * 清理旧日志
     *
     * @param int $days 保留天数
     * @return int 清理的记录数
     */
    public static function clean_old_logs($days = 30)
    {
        global $wpdb;
        $table = self::get_table_name();
        
        if (!self::table_exists()) {
            return 0;
        }
        
        $result = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $days
        ));
        
        if ($result !== false) {
            \error_log("AI Post Debug: 已清理{$result}条超过{$days}天的旧日志");
        }
        
        return $result !== false ? $result : 0;
    }
}
