<?php
// Service/Features/Markdown.php
// 该文件主要负责提供 Markdown 文本到 HTML 格式的转换功能。

namespace HaoZiTeam\AIPost\Service\Features;

defined('ABSPATH') || exit;

class Markdown
{
    /**
     * 将 Markdown 文本转换为 HTML。
     *
     * @param string $markdown Markdown 文本。
     * @return string HTML 格式的文本。
     */
    public static function markdownToHtml($markdown)
    {
        // 兼容性增强：非字符串直接返回空
        if (!is_string($markdown)) {
            return '';
        }

        // 统一换行符，避免 Windows (\r\n) 或旧 Mac (\r) 导致的表格/段落正则失配
        $markdown = str_replace(["\r\n", "\r"], "\n", $markdown);

        // 处理多行代码块（优先处理，完全移除反引号）
        $markdown = preg_replace('/```\s*\w*\s*\n([\s\S]*?)\n```/', '<pre><code>$1</code></pre>', $markdown);
        
        // 处理行内代码块（保留内容但移除反引号）
        $markdown = preg_replace('/`([^`]+)`/', '<code>$1</code>', $markdown);

        // 处理标题标签，例如 # Heading => <h1>Heading</h1>
        $markdown = preg_replace('/^(#{6}) (.*)$/m', '<h6>$2</h6>', $markdown);
        $markdown = preg_replace('/^(#{5}) (.*)$/m', '<h5>$2</h5>', $markdown);
        $markdown = preg_replace('/^(#{4}) (.*)$/m', '<h4>$2</h4>', $markdown);
        $markdown = preg_replace('/^(#{3}) (.*)$/m', '<h3>$2</h3>', $markdown);
        $markdown = preg_replace('/^(#{2}) (.*)$/m', '<h2>$2</h2>', $markdown);
        $markdown = preg_replace('/^(#{1}) (.*)$/m', '<h1>$2</h1>', $markdown);

        // 替换粗体，例如 **text** => <strong>text</strong>
        $markdown = preg_replace('/\*\*(.+?)\*\*/s', '<strong>$1</strong>', $markdown);

        // 替换斜体，例如 *text* => <em>text</em>
        $markdown = preg_replace('/\*(.+?)\*/s', '<em>$1</em>', $markdown);

        // 替换有序列表，例如 1. item => <li>item</li>
        $markdown = preg_replace_callback('/(\d+\.\s.*?)(?:\n(?!\d+\.\s)|$)/s', function ($matches) {
            return preg_replace('/\d+\.\s(.*)/', '<li>$1</li>', $matches[0]);
        }, $markdown);

        // 替换无序列表，例如 - item => <li>item</li>
        $markdown = preg_replace_callback('/(-\s.*?)(?:\n(?!-\s)|$)/s', function ($matches) {
            return preg_replace('/-\s(.*)/', '<li>$1</li>', $matches[0]);
        }, $markdown);

        // 替换链接，例如 [link text](url) => <a href="url">link text</a>
        $markdown = preg_replace('/\[(.+?)\]\((.+?)\)/', '<a href="$2">$1</a>', $markdown);

        // 替换图片，例如 ![alt text](url) => <img src="url" alt="alt text">
        $markdown = preg_replace('/!\[(.*?)\]\((.+?)\)/', '<img src="$2" alt="$1">', $markdown);

        // 添加对Markdown表格的处理
        $markdown = preg_replace_callback(
            '/^\|(.+)\|\s*\n\|(?:[-:]+\|)+\s*\n(\|(?:.+)\|\s*\n)+/m',
            function ($matches) {
                $table = '<table style="width:100%; border-collapse:collapse; margin:15px 0;">';
                $lines = explode("\n", trim($matches[0]));
                
                // 处理表头
                $headerCells = explode('|', trim($lines[0], '|'));
                $table .= '<thead><tr>';
                foreach ($headerCells as $cell) {
                    $table .= '<th style="border:1px solid #ddd; padding:8px; text-align:center; background-color:#f2f2f2;">' . trim($cell) . '</th>';
                }
                $table .= '</tr></thead>';
                
                // 处理表格内容
                $table .= '<tbody>';
                for ($i = 2; $i < count($lines); $i++) {
                    if (empty(trim($lines[$i]))) continue;
                    $rowCells = explode('|', trim($lines[$i], '|'));
                    $table .= '<tr>';
                    foreach ($rowCells as $index => $cell) {
                        $bgColor = ($i % 2 == 0) ? '#f9f9f9' : '#ffffff'; // 交替行背景色
                        $table .= '<td style="border:1px solid #ddd; padding:8px; text-align:center; background-color:' . $bgColor . ';">' . trim($cell) . '</td>';
                    }
                    $table .= '</tr>';
                }
                $table .= '</tbody></table>';
                
                return $table;
            },
            $markdown
        );

        // 添加对分散的Markdown表格的处理（被图片或其他内容打断的表格）
        $markdown = preg_replace_callback(
            '/\|\s*([^\n|]+\|)+\s*\n\|\s*[-:]+(?:\|[-:]+)*\|\s*\n(?:\|\s*([^\n|]+\|)+\s*\n)+/m',
            function ($matches) {
                $table = '<table style="width:100%; border-collapse:collapse; margin:15px 0;">';
                $lines = explode("\n", trim($matches[0]));
                
                // 处理表头
                $headerCells = explode('|', trim($lines[0], '|'));
                $table .= '<thead><tr>';
                foreach ($headerCells as $cell) {
                    $table .= '<th style="border:1px solid #ddd; padding:8px; text-align:center; background-color:#f2f2f2;">' . trim($cell) . '</th>';
                }
                $table .= '</tr></thead>';
                
                // 处理表格内容
                $table .= '<tbody>';
                for ($i = 2; $i < count($lines); $i++) {
                    if (empty(trim($lines[$i]))) continue;
                    $rowCells = explode('|', trim($lines[$i], '|'));
                    $table .= '<tr>';
                    foreach ($rowCells as $index => $cell) {
                        $bgColor = ($i % 2 == 0) ? '#f9f9f9' : '#ffffff'; // 交替行背景色
                        $table .= '<td style="border:1px solid #ddd; padding:8px; text-align:center; background-color:' . $bgColor . ';">' . trim($cell) . '</td>';
                    }
                    $table .= '</tr>';
                }
                $table .= '</tbody></table>';
                
                return $table;
            },
            $markdown
        );

        // 处理更复杂的被打断的表格情况（只有头行和一个数据行的情况）
        $markdown = preg_replace_callback(
            '/\|\s*([^\n|]+\|)+\s*\n\|\s*[-:]+(?:\|[-:]+)*\|\s*\n\|\s*([^\n|]+\|)+\s*\n/m',
            function ($matches) {
                $table = '<table style="width:100%; border-collapse:collapse; margin:15px 0;">';
                $lines = explode("\n", trim($matches[0]));
                
                // 处理表头
                $headerCells = explode('|', trim($lines[0], '|'));
                $table .= '<thead><tr>';
                foreach ($headerCells as $cell) {
                    $table .= '<th style="border:1px solid #ddd; padding:8px; text-align:center; background-color:#f2f2f2;">' . trim($cell) . '</th>';
                }
                $table .= '</tr></thead>';
                
                // 处理表格内容
                $table .= '<tbody>';
                for ($i = 2; $i < count($lines); $i++) {
                    if (empty(trim($lines[$i]))) continue;
                    $rowCells = explode('|', trim($lines[$i], '|'));
                    $table .= '<tr>';
                    foreach ($rowCells as $index => $cell) {
                        $bgColor = ($i % 2 == 0) ? '#f9f9f9' : '#ffffff'; // 交替行背景色
                        $table .= '<td style="border:1px solid #ddd; padding:8px; text-align:center; background-color:' . $bgColor . ';">' . trim($cell) . '</td>';
                    }
                    $table .= '</tr>';
                }
                $table .= '</tbody></table>';
                
                return $table;
            },
            $markdown
        );
        
        // 检测并修复被分隔的表格行
        $pattern = '/\|\s*(\S+[^|\n]*\s*\|)+\s*\n/';
        preg_match_all($pattern, $markdown, $matches, PREG_OFFSET_CAPTURE);
        
        // 如果找到了独立的表格行
        if (!empty($matches[0])) {
            // 按位置倒序处理，避免修改位置导致的问题
            $rowMatches = array_reverse($matches[0]);
            
            foreach ($rowMatches as $match) {
                $row = $match[0];
                $position = $match[1];
                
                // 检查前后文是否可能是表格部分
                $prevChar = $position > 0 ? $markdown[$position - 1] : '';
                $nextChar = $position + strlen($row) < strlen($markdown) ? $markdown[$position + strlen($row)] : '';
                
                // 如果这行前后都没有表格结构标记，可能是被打断的表格行
                if (strpos($prevChar . $nextChar, '|') === false) {
                    // 尝试将这行转换为表格单元格
                    $cells = array_map('trim', explode('|', trim($row, "|\n\r ")));
                    if (count($cells) > 1) {
                        $html = '<table class="ai-recovered-table" style="width:100%; border-collapse:collapse; margin:15px 0;"><tr>';
                        foreach ($cells as $cell) {
                            if (!empty($cell)) {
                                $html .= '<td style="border:1px solid #ddd; padding:8px; text-align:center;">' . $cell . '</td>';
                            }
                        }
                        $html .= '</tr></table>';
                        
                        // 替换原始行
                        $markdown = substr($markdown, 0, $position) . $html . substr($markdown, $position + strlen($row));
                    }
                }
            }
        }

        // 修改：确保段落包裹逻辑跳过常见的HTML块级和表格标签
		$markdown = preg_replace('/^(?!(?:#{1,6}|\d+\.\s-|-\s|<h[1-6]|<p|<div|<ul|<ol|<li|<blockquote|<pre|<hr|<table|<thead|<tbody|<tr|<th|<td|<\/?\w+[^>]*>))(.+)$/im', '<p>$1</p>', $markdown);

        // 移除空的 <h1> 到 <h6> 标签
        $markdown = preg_replace('/ < h[1 - 6] > \s * <\/h[1-6]>/s', '', $markdown);

        // 移除空的 <li></li> 标签
        $markdown = preg_replace('/<li>\s*<\/li>/s', '', $markdown);

        // 移除只包含非内容字符的 <p> 和 <li> 标签
        $markdown = preg_replace('/<p>\s*[-—]\s*<\/p>/s', '', $markdown);
        $markdown = preg_replace('/<li>\s*[-—]\s*<\/li>/s', '', $markdown);

        // 移除多余的 -- 符号
        $markdown = preg_replace('/--+/s', '', $markdown);

        // 移除没有内容的 :
        $markdown = preg_replace('/(?<=^|\s):/', '', $markdown);

        // 标签过滤（移除下面这行以保留 code 和 pre）
        // $markdown = preg_replace('/<\/?(code|pre)>/', '', $markdown);

        // 清理空的 <p> 标签
        $markdown = preg_replace('/<p>\s*<\/p>/s', '', $markdown);

        return $markdown;
    }
} 