<?php
// Service/Features/LanguageUtils.php
// 该文件主要负责提供语言相关的辅助功能，例如语言代码和名称的映射。

namespace HaoZiTeam\AIPost\Service\Features;

// 如果需要 Parsedown，确保 use 语句正确
// use Parsedown; 
// 或者 use \Parsedown; // 如果 Parsedown 在全局命名空间

defined('ABSPATH') || exit;

class LanguageUtils
{
    /**
     * 获取语言代码到显示名称的映射数组。
     *
     * @return array 语言映射数组。
     */
    public static function get_language_map()
    {
        return [
            'tags-zh' => '中文简体',
            'tags-zh-fan' => '中文繁体',
            'tags-en' => '英文',
            'tags-vi' => '越南语',
            'tags-ja' => '日语',
            'tags-ko-KR' => '韩语',
            'tags-sa-IN' => '梵文-印度',
            'tags-pt-BR' => '葡萄牙语',
            'tags-pt-BR-BR' => '巴西葡萄牙语',
            'tags-es' => '西班牙语',
            'custom' => '自定义语言',
        ];
    }

    /**
     * Adds the 'lang' attribute to the HTML tag based on WordPress settings.
     * Applies zh-Hans to zh-CN mapping.
     *
     * @param string $output The default language attributes string.
     * @return string Modified language attributes string.
     */
    public static function add_html_lang_attribute($output)
    {
        $lang = get_bloginfo('language');
        if ($lang) {
            // Apply zh-Hans to zh-CN mapping
            if ($lang === 'zh-Hans') {
                $lang = 'zh-CN';
            }
            // Construct the lang attribute
            // We replace the default output to ensure only our standard format is used
            $new_output = 'lang="' . esc_attr($lang) . '"';
            return $new_output;
        }
        // Return potentially modified or original output if language couldn't be retrieved
        return $output;
    }

    /**
     * 将 Markdown 文本转换为 HTML。
     *
     * @param string $markdownText Markdown 格式的文本。
     * @return string HTML 格式的文本。
     */
    public static function markdownToHtml(string $markdownText): string
    {
        // 确保 Parsedown 类可用
        if (class_exists('Parsedown')) {
            $Parsedown = new \Parsedown(); // 或者 new Parsedown(); 取决于 use 语句
            return $Parsedown->text($markdownText);
        } elseif (class_exists('\\Parsedown')) { // 检查全局命名空间
            $Parsedown = new \Parsedown();
            return $Parsedown->text($markdownText);
        } else {
            // Fallback or error logging if Parsedown is not available
            error_log('AI Post Error: Parsedown class not found. Cannot convert Markdown to HTML.');
            // 返回原始文本或进行简单处理，例如将换行符转为 <br>
            // return nl2br(htmlspecialchars($markdownText)); // 这是一个非常基础的降级处理
            return $markdownText; // 或者直接返回，让问题更明显
        }
    }

    // ... 可能还有 numberToChinese 等其他方法 ...
} 

