<?php
// Service/Features/HtmlUtils.php
// 该文件主要负责提供 HTML 处理相关的辅助功能。

namespace HaoZiTeam\AIPost\Service\Features;

defined('ABSPATH') || exit;

class HtmlUtils
{
    /**
     * 尝试通过在末尾追加 body 和 html 闭合标签来"关闭"HTML 内容。
     * 注意：这种方法通常不推荐用于处理 HTML 片段。
     *
     * @param string $content HTML 内容字符串。
     * @return string 追加了闭合标签的 HTML 内容字符串。
     */
    public static function close_html_tags($content)
    {
        // 简单地在内容末尾追加 body 和 html 闭合标签
        return $content . '</body></html>';
    }

    /**
     * 判断内容是否“实质为空”
     * 规则：
     * - 若包含任意 <img>，则视为非空（避免仅图片内容被判空导致重复生成）
     * - 去除短码、解码HTML实体、移除所有标签后，若纯文本为空，则为空
     */
    public static function is_content_effectively_empty(string $content): bool
    {
        // 快速判空（null/空字符串）
        if ($content === '' || $content === null) {
            return true;
        }

        // 1) 检测是否包含图片标签，若有，直接视为非空
        if (preg_match('/<img\b[^>]*>/i', $content)) {
            return false;
        }

        // 2) 去除短码（如果 WordPress 方法可用则调用）
        if (function_exists('strip_shortcodes')) {
            $content_no_shortcodes = strip_shortcodes($content);
        } else {
            // 简单回退：去掉形如 [shortcode] 的片段
            $content_no_shortcodes = preg_replace('/\[[^\]]*\]/', '', $content);
        }

        // 3) 解码 HTML 实体（含 &nbsp;）并统一空白
        $decoded = html_entity_decode($content_no_shortcodes, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // 4) 去除所有标签后做 trim
        $plain = trim(strip_tags($decoded));

        // 5) 折叠多余空白
        $plain = preg_replace('/\s+/u', ' ', $plain);
        $plain = trim($plain);

        return ($plain === '');
    }

    /**
     * 检查并处理空标题
     * @param string $title 原始标题
     * @param string $content 文章内容
     * @param string $mode 处理方式：abort/auto/random
     * @return string|false 返回最终标题，或abort时返回false
     */
    public static function check_and_handle_empty_title($title, $content, $mode = 'abort')
    {
        $title = trim($title);
        if ($title !== '') {
            return $title;
        }
        switch ($mode) {
            case 'abort':
                // 直接中止
                return false;
            case 'auto':
                // 尝试从内容中提取前30字作为标题
                $auto_title = mb_substr(strip_tags($content), 0, 30);
                if ($auto_title === '') {
                    $auto_title = '未命名文章-' . date('Ymd-His');
                }
                return $auto_title;
            case 'random':
                // 用随机字符串填充
                return '文章-' . substr(md5(uniqid(mt_rand(), true)), 0, 8);
            default:
                // 默认中止
                return false;
        }
    }
} 